---Helper library to create touchscreen-like screens. Feel free to copy it to your projects if needed,
---it’s CC0 and everything.
---Note: for it to work properly, use `touchscreen.tapped()` instead of regular `ui…` clicks.
touchscreen = {}

local sleep = 0
local mouseDown = false
local mouseClicked = false
local tappedBase = false
local tapped = false
local prevMousePosY = -1
local mouseDownDelta = 0
local scrollingVelocity = 0
local doubleTappedNow = false
local doubleTappedNext = false
local scrollBarVisible = 0
local vkPosition = 1e9
local vkActiveText = nil
local vkHovered = false
local vkActive = 0
local vkCooldown = 0
local vkHeldTimer = nil

---Call this function at the start of a frame.
function touchscreen.update(forceAwake, dt)
  if vkActiveText then
    forceAwake = true
  end
  if scrollBarVisible <= 0 and not ui.anyItemActive() and not ui.isAnyMouseDown() and ui.mouseDelta().x == 0 and not forceAwake then
    sleep = sleep + dt
    if sleep > 3 then
      ac.skipFrame()
      return true
    end
  else
    if forceAwake then
      ac.boostFrameRate()
    end
    sleep = 0
  end

  tappedBase = false
  doubleTappedNow = false
  mouseClicked = false

  if scrollBarVisible > 0 then
    scrollBarVisible = scrollBarVisible - dt
  end
  if ui.mouseDown() ~= mouseDown then
    mouseDown = not mouseDown
    mouseClicked = mouseDown
    prevMousePosY = ui.mousePos().y
    if mouseDown then 
      scrollingVelocity = 0 
      mouseDownDelta = 0
    else
      tappedBase = mouseDownDelta < 10
      doubleTappedNow = tappedBase and doubleTappedNext
      doubleTappedNext = false
    end
  elseif mouseDown then
    scrollingVelocity = vkHovered and 0 or prevMousePosY - ui.mousePos().y
    prevMousePosY = ui.mousePos().y
    mouseDownDelta = mouseDownDelta + #ui.mouseDelta()
    if math.abs(scrollingVelocity) > 0.1 then
      scrollBarVisible = 2
      ac.boostFrameRate()
    end
  else
    scrollingVelocity = scrollingVelocity * 0.9
  end

  if ui.mouseDoubleClicked() then
    doubleTappedNext = true
  end

  if vkActive > 1 then 
    vkActive = vkActive - 1
  else
    if vkCooldown > 0 then vkCooldown = vkCooldown - dt end
    if vkActive > 0.999 and not vkActiveText then vkCooldown = 0.5 end
    vkActive = math.applyLag(vkActive, vkCooldown <= 0 and vkActiveText and 1 or 0, 0.8, dt) 
  end
  vkActiveText = nil

  vkPosition = math.ceil(ui.windowHeight() * (1 - math.saturateN(vkActive) * 0.64))
  vkHovered = ui.mousePos().y > vkPosition
  tapped = tappedBase and not vkHovered and vkCooldown <= 0
  if vkHeldTimer and not ui.mouseDown() then
    clearInterval(vkHeldTimer)
    vkHeldTimer = nil
  end
end

local reactivateText = nil
local prevTextField = nil
local charToAdd = nil
local vkBgColor = rgbm.new('#273238')
local vkBtnColor = rgbm.new('#414A51')
local vkBtnPressedColor = rgbm.new('#313A41')
local vkBtnActiveColor = rgbm.new('#6EACA7')
local vkBtnActivePressedColor = rgbm.new('#5E9C97')
local vkBtnShadowColor = rgbm.new('#111A21')
local vkBtnWidth = 70
local vkEnterIcon = ui.Icons.Enter
local vkShift = false
local vkButtonsReady
local vkLayouts = {
  base = {
    { { 'q', '1', 'Q' }, { 'w', '2', 'W' }, { 'e', '3', 'E' }, { 'r', '4', 'R' }, { 't', '5', 'T' }, { 'y', '6', 'Y' }, { 'u', '7', 'U' }, { 'i', '8', 'I' }, { 'o', '9', 'O' }, { 'p', '0', 'P' } },
    { { 'a', '@', 'A' }, { 's', '#', 'S' }, { 'd', '$', 'D' }, { 'f', '_', 'F' }, { 'g', '&', 'G' }, { 'h', '-', 'H' }, { 'j', '+', 'J' }, { 'k', '(', 'K' }, { 'l', ')', 'L' } },
    { { i = ui.Icons.Shift, s = true, w = 1.5, p = vec2(37, 11) }, { 'z', '*', 'Z' }, { 'x', '"', 'X' }, { 'c', '\'', 'C' }, { 'v', ':', 'V' }, { 'b', ';', 'B' }, { 'n', '!', 'N' }, { 'm', '?', 'M' }, { c = ui.KeyIndex.Back, i = ui.Icons.Backspace, w = 1.5 } },
    { { '?123', l = 'symbols', w = 1.5 }, { ',' }, { ' ', w = 5 }, { '.' }, { i = ui.Icons.Enter, c = ui.KeyIndex.Return, p = vec2(33, 13), w = 1.5 } },
  },
  symbols = {
    { { '1' }, { '2' }, { '3' }, { '4' }, { '5' }, { '6' }, { '7' }, { '8' }, { '9' }, { '0' } },
    { { '@' }, { '#' }, { '$' }, { '%' }, { '&' }, { '-' }, { '+' }, { '(' }, { ')' } },
    { { '=\\<', l = 'symbolsAlt', w = 1.5, p = vec2(37, 11) }, { '*' }, { '"' }, { '\'' }, { ':' }, { ';' }, { '!' }, { '?' }, { c = ui.KeyIndex.Back, i = ui.Icons.Backspace, w = 1.5 } },
    { { 'ABC', l = 'base', w = 1.5 }, { ',' }, { '_' }, { ' ', w = 3 }, { '/' }, { '.' }, { i = ui.Icons.Enter, c = ui.KeyIndex.Return, p = vec2(33, 13), w = 1.5 } },
  },
  symbolsAlt = {
    { { '~' }, { '`' }, { '|' }, { '•' }, { '√' }, { 'π' }, { '÷' }, { '×' }, { '¶' }, { '∆' } },
    { { '£' }, { '¢' }, { '€' }, { '¥' }, { '∧' }, { '°' }, { '=' }, { '{' }, { '}' } },
    { { '?123', l = 'symbols', w = 1.5, p = vec2(37, 11) }, { '\\' }, { '©' }, { '®' }, { '™' }, { '‰' }, { '[' }, { ']' }, { c = ui.KeyIndex.Back, i = ui.Icons.Backspace, w = 1.5 } },
    { { 'ABC', l = 'base', w = 1.5 }, { ',' }, { '<' }, { ' ', w = 3 }, { '>' }, { '.' }, { i = ui.Icons.Enter, c = ui.KeyIndex.Return, p = vec2(33, 13), w = 1.5 } },
  }
}
local vkLayout = vkLayouts.base

local function keyboardButtonLayout(cur, char)
  local w = (char.w or 1) * vkBtnWidth
  char.p1 = cur:clone()
  char.p2 = cur + vec2(w - 10, 46)
  char.s1 = char.p1 + vec2(0, 2)
  char.s2 = char.p2 + vec2(0, 2)
  if char.i then char.ip = char.p1 + (char.p or vec2(33, 11)) end
  if char[1] then char.t1 = char.p1 + vec2((w - 10) / 2 - ui.measureDWriteText(char[1], 24).x / 2, 5) end
  if char[3] then char.t3 = char.p1 + vec2((w - 10) / 2 - ui.measureDWriteText(char[3], 24).x / 2, 5) end
  if char[2] then 
    char.t2 = char.p1 + vec2((w - 10) / 2 - ui.measureDWriteText(char[2], 24).x / 2, -50)
    char.u1 = char.p1 + vec2(w - 20 - ui.measureDWriteText(char[2], 14).x / 2, 0) 
  end
  return w
end

local function keyboardButton(char, layer)
  local cur = char.p1
  local p2 = char.p2
  if layer == 1 then
    local enterButton = char.c == ui.KeyIndex.Return
    ui.drawRectFilled(char.s1, char.s2, vkBtnShadowColor, 8)
    ui.drawRectFilled(cur, p2, (char.s and vkShift == 2 or enterButton) and (char.hovered and vkBtnActivePressedColor or vkBtnActiveColor) 
      or char.hovered and vkBtnPressedColor or vkBtnColor, 8)

    if mouseClicked and ui.rectHovered(cur, p2) then
      char.hovered = true
      char.popup = false
      vkHeldTimer = char.c == ui.KeyIndex.Back
        and setInterval(function () charToAdd = ui.KeyIndex.Back end, 0.12)
        or char[2] and setTimeout(function () char.popup, vkHeldTimer = true, nil end, 0.3)
    elseif char.hovered and not mouseDown then
      if ui.rectHovered(cur, p2) then
        if char.s then
          vkShift = vkShift == true and 2 or not vkShift
        elseif char.l then
          vkLayout = vkLayouts[char.l]
        else
          charToAdd = char.c or (char.popup and char[2] or vkShift and char[3] or char[1])
          if vkShift == true then vkShift = false end
        end
      end
      char.hovered, char.popup = false, false
    end

    if char.i then
      ui.setCursor(char.ip)
      ui.icon24(enterButton and vkEnterIcon or char.s and vkShift and ui.Icons.ShiftActive or char.i, 24, rgbm.colors.white)
    end
  end

  if layer == 2 and char[1] then
    ui.dwriteDrawText(vkShift and char[3] or char[1], 24, vkShift and char.t3 or char.t1, rgbm.colors.white)
    if char[2] then
      ui.dwriteDrawText(char[2], 14, char.u1, rgbm(1, 1, 1, 0.5))
    end
  end

  if layer == 3 then
    ui.pushClipRectFullScreen()
    ui.drawRectFilled(cur - vec2(0, 60), p2, vkBtnShadowColor, 8)
    ui.dwriteDrawText(char[2], 24, char.t2, rgbm.colors.white)
    ui.popClipRect()
  end
end

local function updateKeyboardButtons()
  local r = {}
  for i = 1, #vkLayout do
    local cur = vec2(140 + (i == 2 and vkBtnWidth/2 or 0), 55 * (i - 1) + 10)
    for j = 1, #vkLayout[i] do
      cur.x = cur.x + keyboardButtonLayout(cur, vkLayout[i][j])
      table.insert(r, vkLayout[i][j])
    end
  end
  return r
end

---Call this function at the end of a frame, it would draw an on-screen keyboard if needed.
function touchscreen.keyboard()
  if vkActiveText ~= nil then
    prevTextField = vkActiveText
    vkEnterIcon = vkActiveText == 'Search' and ui.Icons.Search or ui.Icons.Enter
  end
  if not vkButtonsReady or vkButtonsReady.layout ~= vkLayout then
    vkButtonsReady = updateKeyboardButtons()
  end
  if vkActive > 0.001 then
    ui.setCursor(vec2(0, vkPosition))
    ui.childWindow('onscreenKeyboard', ui.availableSpace() + vec2(0, 10), true, ui.WindowFlags.NoFocusOnAppearing, function ()
      ui.setCursor(0)
      ui.drawRectFilled(0, ui.availableSpace(), vkBgColor)

      local btnPopup
      for i = 1, #vkButtonsReady do
        local btn = vkButtonsReady[i]
        keyboardButton(btn, 1)
        if btn.popup then btnPopup = btn end
      end
      for i = 1, #vkButtonsReady do
        keyboardButton(vkButtonsReady[i], 2)
      end
      if btnPopup then
        keyboardButton(btnPopup, 3)
      end

      if ui.windowHovered(ui.HoveredFlags.AllowWhenBlockedByActiveItem) and (ui.mouseClicked() or ui.mouseReleased()) then
        reactivateText = prevTextField
        vkActive = 3
      end
    end)
  end
end

function touchscreen.keyboardHovered()
  return vkHovered
end

---Text input control with onscreen keyboard support. Returns updated string (which would be the input string unless it changed, so no)
---copying there. Second return value would change to `true` when text has changed. Example:
---```
---myText = ui.inputText('Enter something:', myText)
---```
---@param label string
---@param str string
---@param flags ui.InputTextFlags
---@return string
---@return boolean
function touchscreen.inputText(label, str, flags)
  local done = false
  if label == reactivateText then
    reactivateText = nil
    ui.setKeyboardFocusHere()
  elseif charToAdd ~= nil then
    if type(charToAdd) == 'number' then
      if charToAdd == ui.KeyIndex.Return then done = true end
      ui.setKeyboardButtonDown(charToAdd)
    else 
      ui.addInputCharacter(charToAdd:byte())
    end
    charToAdd = nil
  end

  local ret = ui.inputText(label, str, bit.bor(flags, ui.InputTextFlags.NoUndoRedo))
  if ui.itemActive() then
    vkActiveText = label
  end
  if str and #str > 0 then
    local c = ui.getCursor()
    ui.setItemAllowOverlap()
    ui.sameLine(0, 0)
    ui.offsetCursorX(-32)
    if touchscreen.iconButton(ui.Icons.Cancel, vec2(32, 20), 0.3, nil, 0.4) then
      ret = ''
    end
    if ui.itemClicked() then
      vkActiveText = nil
    end
    ui.setCursor(c)
  end
  return ret, done
end

---Adds touch scrolling to current list. Call it from a scrollable window.
function touchscreen.scrolling()
  ui.setScrollY(scrollingVelocity, true)

  local scrollMax = ui.getScrollMaxY()
  if scrollMax > 1 then
    local window = ui.windowSize()
    local scrollY = ui.getScrollY()
    local barMarginX = 4
    local barMarginY = 4
    local barArea = window.y - barMarginY * 2
    local barX = window.x - math.lerp(-2, barMarginX, math.saturateN(scrollBarVisible * 4))
    local barHeight = math.lerp(40, barArea, window.y / (scrollMax + window.y))
    local barY = scrollY + barMarginY + scrollY / scrollMax * (barArea - barHeight)
    ui.setCursor(0)
    ui.drawLine(vec2(barX, barY), vec2(barX, barY + barHeight), rgbm.colors.white, 2)
  end
end

---Returns true if screen was just tapped
function touchscreen.tapped()
  return tapped
end

---Returns true if screen was just double tapped
function touchscreen.doubleTapped()
  return doubleTappedNow
end

---Returns true if previous item was just tapped
function touchscreen.itemTapped()
  return tapped and ui.itemHovered()
end

local defBaseSize = vec2(24, 24)
local defSize = vec2(24, 24)

---Touch button with an icon
---@param icon ui.Icons
---@param size vec2
---@param iconAlphaOrColor number|rgbm
---@param iconAngle number|nil
---@param iconScale number|nil
---@return boolean
function touchscreen.iconButton(icon, size, iconAlphaOrColor, iconAngle, iconScale)
  local p = ui.getCursor()
  if not vec2.isvec2(size) then size = defBaseSize:set(size or 40, size or 40) end
  ui.offsetCursorX((size.x - defSize.x) / 2)
  ui.offsetCursorY((size.y - defSize.y) / 2)
  if iconAngle ~= nil then ui.beginRotation() end
  if iconScale ~= nil then ui.beginScale() end
  if type(iconAlphaOrColor) == 'number' then ui.pushStyleVarAlpha(iconAlphaOrColor) end
  ui.icon24(icon, defSize, rgbm.isrgbm(iconAlphaOrColor) and iconAlphaOrColor or nil)
  if iconAngle ~= nil then ui.endRotation(iconAngle) end
  if iconScale ~= nil then ui.endScale(iconScale) end
  if type(iconAlphaOrColor) == 'number' then ui.popStyleVar() end
  ui.setCursor(p)
  ui.invisibleButton(icon, size)
  return tapped and ui.itemHovered()
end

local iconSizeValue = vec2(24, 24)

---Touch button with an icon
---@param label string
---@param buttonSize vec2
---@param buttonColor rgbm
---@param fontSize number
---@param icon ui.Icons
---@param iconSize vec2
---@return boolean
function touchscreen.button(label, buttonSize, buttonColor, fontSize, icon, iconSize)
  local p = ui.getCursor()
  ui.invisibleButton(label, buttonSize)
  ui.drawRectFilled(p, p + buttonSize, 
    ui.itemActive() and rgbm(buttonColor.r * 0.8, buttonColor.g * 0.8, buttonColor.b * 0.8, buttonColor.mult) or buttonColor)
  ui.setCursor(p)
  if fontSize ~= 0 then
    ui.dwriteTextAligned(label, fontSize, ui.Alignment.Center, ui.Alignment.Center, buttonSize)
  end
  if icon ~= nil then
    if iconSize then iconSizeValue:set(iconSize) end
    ui.offsetCursorX((buttonSize.x - iconSizeValue.x) / 2)
    ui.offsetCursorY((buttonSize.y - iconSizeValue.y) / 2)
    ui.icon24(icon, iconSizeValue)
  end
  return tapped and ui.itemHovered()
end

local loadingSize = vec2()

local function step(v)
  return (math.floor(v / 2) + math.smoothstep(math.min(v % 2, 1)))
end

function touchscreen.loading(size)
  local s = loadingSize:set(size)
  local t = ui.time()
  local r = math.min(s.x, s.y) / 2
  ui.pathArcTo(ui.getCursor() + s * 0.5, r, step(t * 1.3 + 1) * 4.5 + t * 3, step(t * 1.3) * 4.5 + 5 + t * 3, 40)
  ui.pathStroke(rgbm.colors.white, false, r / 5)
  ui.dummy(s)
end
